/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using Borland.Eco.DataRepresentation;
using Borland.Eco.UmlRt;

namespace Borland.Eco.Cache
{

	public sealed class LocatorEventArgs: EventArgs
	{
		private readonly Locator  m_Locator;

		public  Locator Locator
		{
			get {return m_Locator;}
		}

		public LocatorEventArgs(Locator locator)
		{
			m_Locator = locator;
		}
	}
	public delegate void ValueChangedDelegate(Locator locator, IStructuralFeature feature);
	public delegate void LocatorEventHandler(object sender, LocatorEventArgs args);

	public class CanDestroyResult
	{
		private bool result = true;
		public bool Result
		{
			get { return result; }
			set { result = value; }
		}
	}

	public delegate void CanDestroyDelegate(Locator locator, CanDestroyResult result);

	/// <summary>
	/// Whether or not the object exists.
	/// </summary>
	public enum ExistenceState
	{
		NotCreated,
		Existing,
		Deleted
	}

	/// <summary>
	/// How the current value in memory is related to the database. Current: they are believed to be the same.
	/// Modified: The value has been changed in memory, but not yet saved. Invalid: The value needs to be fetched from Db.
	/// Transient: The value in memory has no equivalence in Db.
	/// </summary>
	public enum PersistenceState
	{
		Current,
		Modified,
		Invalid,
		Transient
	}

	/// <summary>
	/// A locator is an identifier that's used to identify an object. It has reference semantics.
	/// It has no features of its own, but is accessed thorugh the ICache interface
	/// </summary>
	public abstract class Locator
	{
	}

	/// <summary>
	/// The function of an ICache is to store data about objects in memory, in relation to the Db.
	/// </summary>
	public interface ICache
	{
		/// <summary>
		/// The locators of all the objects in the cache
		/// </summary>
		Locator[] GetAllLocators();
		/// <summary>
		/// Changes the ObjectIds of objects in the cache according to to translations in the list
		/// </summary>
		void ApplyTranslationList(IdTranslationList idTranslationList);
		/// <summary>
		/// Changes the ObjectId of an object to newId
		/// </summary>
        void SetObjectId(Locator locator, ObjectId newId);
		/// <summary>
		/// Called whenever the current value of an attribute or association end changes
		/// If the persistencestate is invalid, no event is sent. If it changes from current or modified to invalid,
		///	an event is sent.
		/// </summary>
		event ValueChangedDelegate ValueChanged;
		/// <summary>
		/// Called when the existence state of an object changes
		/// </summary>
		event LocatorEventHandler ExistenceChanged;
		/// <summary>
		/// Called when a new locator gets created
		/// </summary>
		event LocatorEventHandler LocatorAdded;
		/// <summary>
		/// Called when the dirtyness of an object may have changed
		/// </summary>
		event LocatorEventHandler DirtyChanged;
		event CanDestroyDelegate CanDestroy;
		/// <summary>
		/// Returns the locator associated with a particular ObjectId. If it doesn't exist in the cache, it is created
		/// </summary>
		Locator GetEnsuredLocatorById(ObjectId objectId);
		/// <summary>
		/// Returns the locator associated with a particular ObjectId. If it doesn't exist in the cache, null is returned
		/// </summary>
		Locator GetLocatorById(ObjectId objectId);
		/// <summary>
		/// Returns the ObjectId associated with a locator
		/// </summary>
		ObjectId GetIdByLocator(Locator locator);
		/// <summary>
		/// Creates, and returns, a new locator with the UmlClass class_. A non-persistent id will be generated and assigned automatically.
		/// </summary>
		Locator CreateNewObject(IClass class_);
		/// <summary>
		/// Remove locator from the cache.
		/// </summary>
		void DestroyObject(Locator locator);
		/// <summary>
		/// Returns the IClass of the object
		/// </summary>
		IClass GetUmlClass(Locator locator);
		/// <summary>
		/// Sets the existence state of the object
		/// </summary>
		void SetExistenceState(Locator locator, ExistenceState value);
		/// <summary>
		/// Returns the existence state of the object
		/// </summary>
		ExistenceState GetExistenceState(Locator locator);
		/// <summary>
		/// Sets the persistence state of the object. It indicates the exisitence state only, not the members of the object
		/// </summary>
		void SetPersistenceState(Locator locator, PersistenceState value);
		/// <summary>
		/// Returns the persistence state of the object. It indicates the persistence state only, not the members of the object
		/// </summary>
		PersistenceState GetPersistenceState(Locator locator);
		/// <summary>
		/// Returns true if the object referred by the locator is persistent, false if it's transient
		/// </summary>
		bool IsPersistent(Locator locator);
		/// <summary>
		/// Returns true if the object referred by the locator is dirty, i.e.
		/// if it is persistent, and either is nwe, or delteted, or had at least one
		/// member that is Modified
		/// </summary>
		bool IsDirty(Locator locator);
		///<summary>
		/// Returns true if the locator belongs to the cache
		///</summary>
		bool Contains(Locator locator);
		///<summary>
		/// If this object uses timestamp-based optimistic locking, this is the timestamp of the object. Otherwise it is -1
		///</summary>
		int GetTimestamp(Locator locator);
		///<summary>
		/// Sets the timestamp.
		/// If this object uses timestamp-based optimistic locking, this is the timestamp of the object. Otherwise it is -1
		///</summary>
		void  SetTimestamp(Locator locator, int value);
		///<summary>
		/// Sets the order number of an item in an ordered association.
		/// The structuralFeature feature must refer to an embedded single end of an ordered association.
		/// The correponding multi end will be sorted automatically.
		///</summary>
		void SetCurrentOrderNo(Locator locator, IStructuralFeature structuralFeature, int orderNo);
		///<summary>
		/// Same as SetCurrentOrderNo, except it sets both the current value and the fetched value
		///</summary>
		void SetCurrentAndFetchedOrderNo(Locator locator, IStructuralFeature structuralFeature, int orderNo);
		///<summary>
		/// Returns the order number of an item in an ordered association.
		///</summary>
		int GetCurrentOrderNo(Locator locator, IStructuralFeature structuralFeature);
		///<summary>
		/// Returns the persistence state of a member
		///</summary>
		PersistenceState GetMemberPersistenceState(Locator locator, IStructuralFeature structuralFeature);
		///<summary>
		/// Sets the persistence state of a member
		///</summary>
		void SetMemberPersistenceState(Locator locator, IStructuralFeature structuralFeature, PersistenceState value);
		///<summary>
		/// Returns the current value of a member
		///</summary>
		object GetCurrentValue(Locator locator, IStructuralFeature structuralFeature);
		///<summary>
		/// Sets the current value of a member
		///</summary>
		void SetCurrentValue(Locator locator, IStructuralFeature structuralFeature, object value);
		///<summary>
		/// Returns the fetched value of a member
		///</summary>
		object GetFetchedValue(Locator locator, IStructuralFeature structuralFeature);
		///<summary>
		/// Returns true if the fetched value differs from the current value
		///</summary>
		bool GetMemberHasFetchedValue(Locator locator, IStructuralFeature structuralFeature);
		///<summary>
		/// Returns the fetched order number of an item in an ordered association
		///</summary>
		int GetFetchedOrderNo(Locator locator, IStructuralFeature structuralFeature);
		///<summary>
		/// Sets the fetched value of a member
		///</summary>
		void SetFetchedValue(Locator locator, IStructuralFeature structuralFeature, object value);
		///<summary>
		/// Sets the current and fetched values of a member. Equvalent to calling SetCurrentValue and SetFetchedValue with the same value.
		///</summary>
		void SetCurrentAndFetchedValue(Locator locator, IStructuralFeature structuralFeature, object value);

	}

}
